#!/usr/bin/env python3
# F16 — Horizon (Reachability NoCommit) — v2
# Present-act engine (stdlib only). Control is pure boolean/ordinal; neighbor-only; NO RNG in control.
# Goal: certify a hard "NoCommit" horizon at radius R_h:
#   - Rays with impact b > R_h must REACH the detector.
#   - Rays with impact b ≤ R_h must FAIL to reach (stuck after first entry into r ≤ R_h).
#   - Boundary is monotone (no holes), with small edge blur and mesh agreement.

import argparse, csv, hashlib, json, math, os, sys
from datetime import datetime, timezone
from typing import List, Dict, Tuple, Optional

def utc_timestamp() -> str:
    return datetime.now(timezone.utc).strftime("%Y-%m-%dT%H-%M-%SZ")

def ensure_dirs(root: str, subs: List[str]) -> None:
    for s in subs: os.makedirs(os.path.join(root, s), exist_ok=True)

def write_text(path: str, txt: str) -> None:
    with open(path, "w", encoding="utf-8") as f: f.write(txt)

def json_dump(path: str, obj: dict) -> None:
    with open(path, "w", encoding="utf-8") as f: json.dump(obj, f, indent=2, sort_keys=True)

def sha256_file(path: str) -> str:
    h = hashlib.sha256()
    with open(path, "rb") as f:
        for chunk in iter(lambda: f.read(1 << 20), b""): h.update(chunk)
    return h.hexdigest()

def isqrt(n: int) -> int:
    return int(math.isqrt(n))

# Optional outer slow-down (present-act, integer-only) as in E14:
# P(r) = 1 + floor(kappa / r) outside the horizon; INSIDE the horizon, commits are DISALLOWED.
def period_outside_horizon(r: int, kappa: int) -> int:
    r = max(1, int(r))
    return 1 + (kappa // r)

def simulate_ray(N:int, cx:int, cy:int, x0:int, x1:int, b:int, Rh:int, kappa:int) -> dict:
    """
    Deterministic east-going ray at y = cy + b (or symmetric if out of bounds).
    Control: at each tick t,
      - r = floor(sqrt((x-cx)^2 + (y-cy)^2))
      - if r <= Rh: E-step is FORBIDDEN (NoCommit region)
      - else E-step allowed only when (t % P_outside(r) == 0)
    No vertical moves; ticks advance regardless (pure present-act gating).
    """
    y = cy + b
    if y < 0 or y >= N:
        y = cy - b
        if y < 0 or y >= N:
            return {"b_shells": b, "reached": False, "ticks": 0,
                    "first_inside_tick": None, "east_steps_after_inside": 0,
                    "east_steps_total": 0, "guard_exceeded": False}

    x = x0
    t = 0
    first_inside_tick: Optional[int] = None
    east_total = 0
    east_after_inside = 0

    # Guard: worst-case step interval outside horizon is <= 1 + kappa
    length = (x1 - x0)
    guard = length * (2 + max(1, kappa))

    while x < x1 and t < guard:
        dx = x - cx
        dy = y - cy
        r = isqrt(dx*dx + dy*dy)

        inside = (r <= Rh)
        if inside and first_inside_tick is None:
            first_inside_tick = t

        if not inside:
            P = period_outside_horizon(r, kappa)
            if (t % P) == 0:
                x += 1
                east_total += 1
        # inside ⇒ NoCommit (no east step)
        if first_inside_tick is not None and x < x1:
            pass
        t += 1

    reached = (x >= x1)
    guard_exceeded = (not reached and t >= guard)

    return {"b_shells": b, "reached": reached, "ticks": t,
            "first_inside_tick": first_inside_tick,
            "east_steps_after_inside": east_after_inside,
            "east_steps_total": east_total,
            "guard_exceeded": guard_exceeded}

def run_panel(manifest: dict, gridN:int, outdir:str, tag:str) -> dict:
    N = gridN
    cx = int(manifest["grid"].get("cx", N//2))
    cy = int(manifest["grid"].get("cy", N//2))
    x_margin = int(manifest["source"].get("x_margin", 16))
    x0, x1 = x_margin, N - x_margin

    Rh = int(manifest["horizon"]["radius_shells"])
    kappa = int(manifest["schedule"].get("kappa", 0))  # 0 ⇒ unit rate outside

    b_list = sorted([int(b) for b in manifest["source"]["impact_params_shells"]])

    rows = []
    for b in b_list:
        r = simulate_ray(N, cx, cy, x0, x1, b, Rh, kappa)
        rows.append(r)

    mpath = os.path.join(outdir, f"f16_{tag}_per_ray.csv")
    with open(mpath, "w", newline="", encoding="utf-8") as f:
        w = csv.writer(f)
        w.writerow(["b_shells","reached","ticks","first_inside_tick","east_steps_after_inside","east_steps_total","guard_exceeded"])
        for r in rows:
            w.writerow([
                r["b_shells"], int(r["reached"]), r["ticks"],
                (r["first_inside_tick"] if r["first_inside_tick"] is not None else ""),
                r["east_steps_after_inside"], r["east_steps_total"], int(r["guard_exceeded"])
            ])

    passes = [r["b_shells"] for r in rows if r["reached"]]
    fails  = [r["b_shells"] for r in rows if not r["reached"]]
    have_both = (len(passes) > 0 and len(fails) > 0)
    b_min_pass = min(passes) if passes else None
    b_max_fail = max(fails)  if fails  else None

    holes = 0
    if have_both:
        for r in rows:
            b = r["b_shells"]
            if r["reached"] and b_max_fail is not None and b <= b_max_fail: holes += 1
            if (not r["reached"]) and b_min_pass is not None and b >= b_min_pass: holes += 1

    edge_blur = None
    horizon_est = None
    if have_both:
        horizon_est = b_min_pass
        edge_blur = b_min_pass - b_max_fail

    interior_ok = True
    for r in rows:
        if not r["reached"]:
            if r["first_inside_tick"] is None or r["east_steps_after_inside"] != 0:
                interior_ok = False
                break

    return {"csv": mpath, "rows": rows, "passes": passes, "fails": fails,
            "have_both": have_both, "b_min_pass": b_min_pass, "b_max_fail": b_max_fail,
            "horizon_est": horizon_est, "edge_blur": edge_blur,
            "holes": holes, "interior_ok": interior_ok}

def main():
    ap = argparse.ArgumentParser()
    ap.add_argument("--manifest", required=True)
    ap.add_argument("--outdir", required=True)
    args = ap.parse_args()

    root = os.path.abspath(args.outdir)
    ensure_dirs(root, ["config","outputs/metrics","outputs/audits","outputs/run_info","logs"])

    with open(args.manifest, "r", encoding="utf-8") as f:
        manifest = json.load(f)
    manifest_path = os.path.join(root, "config", "manifest_f16_v2.json")
    json_dump(manifest_path, manifest)

    write_text(os.path.join(root, "logs", "env.txt"),
               "\n".join([f"utc={utc_timestamp()}",
                          f"os={os.name}", f"cwd={os.getcwd()}",
                          f"python={sys.version.split()[0]}"]))

    N_coarse = int(manifest["grid"]["N"])
    N_fine   = int(manifest["mesh"].get("N_fine", 192))

    coarse = run_panel(manifest, N_coarse, os.path.join(root,"outputs/metrics"), "coarse")
    fine   = run_panel(manifest, N_fine,   os.path.join(root,"outputs/metrics"), "fine")

    holes_max = int(manifest["acceptance"].get("holes_max", 0))
    edge_blur_max = int(manifest["acceptance"].get("edge_blur_max_shells", 1))
    mesh_delta_b_max = int(manifest["mesh"].get("delta_horizon_b_max", 1))
    require_both = bool(manifest["acceptance"].get("require_pass_and_fail", True))
    require_interior_ok = bool(manifest["acceptance"].get("require_interior_neutrality", True))

    mesh_ok = (coarse["horizon_est"] is not None and fine["horizon_est"] is not None and
               abs(coarse["horizon_est"] - fine["horizon_est"]) <= mesh_delta_b_max)

    edge_ok_c = (coarse["edge_blur"] is not None and coarse["edge_blur"] <= edge_blur_max)
    edge_ok_f = (fine["edge_blur"]   is not None and fine["edge_blur"]   <= edge_blur_max)
    holes_ok_c = (coarse["holes"] <= holes_max)
    holes_ok_f = (fine["holes"]   <= holes_max)
    have_both_ok_c = (not require_both) or coarse["have_both"]
    have_both_ok_f = (not require_both) or fine["have_both"]
    interior_ok_c = (not require_interior_ok) or coarse["interior_ok"]
    interior_ok_f = (not require_interior_ok) or fine["interior_ok"]

    passed = bool(mesh_ok and edge_ok_c and edge_ok_f and holes_ok_c and holes_ok_f
                  and have_both_ok_c and have_both_ok_f and interior_ok_c and interior_ok_f)

    audit = {
        "sim": "F16_horizon_v2",
        "coarse": {
            "csv": coarse["csv"], "horizon_est": coarse["horizon_est"],
            "b_min_pass": coarse["b_min_pass"], "b_max_fail": coarse["b_max_fail"],
            "edge_blur": coarse["edge_blur"], "holes": coarse["holes"],
            "have_both": coarse["have_both"], "interior_ok": coarse["interior_ok"]
        },
        "fine": {
            "csv": fine["csv"], "horizon_est": fine["horizon_est"],
            "b_min_pass": fine["b_min_pass"], "b_max_fail": fine["b_max_fail"],
            "edge_blur": fine["edge_blur"], "holes": fine["holes"],
            "have_both": fine["have_both"], "interior_ok": fine["interior_ok"]
        },
        "mesh": {
            "delta_horizon_b": (abs(coarse["horizon_est"] - fine["horizon_est"])
                                if (coarse["horizon_est"] is not None and fine["horizon_est"] is not None) else None),
            "ok": mesh_ok
        },
        "accept": {
            "holes_max": holes_max, "edge_blur_max_shells": edge_blur_max,
            "delta_horizon_b_max": mesh_delta_b_max,
            "require_pass_and_fail": require_both,
            "require_interior_neutrality": require_interior_ok
        },
        "pass": passed,
        "manifest_hash": sha256_file(manifest_path)
    }
    json_dump(os.path.join(root, "outputs", "audits", "f16_audit.json"), audit)

    result_line = (
        "F16_v2 PASS={p} b_h_coarse={bhc} b_h_fine={bhf} Δb={db} "
        "holes(c/f)={hc}/{hf} edge(c/f)={ec}/{ef} interior(c/f)={ic}/{if_} mesh_ok={mk}"
        .format(
            p=passed,
            bhc=(coarse['horizon_est'] if coarse['horizon_est'] is not None else "NA"),
            bhf=(fine['horizon_est']   if fine['horizon_est']   is not None else "NA"),
            db=(abs(coarse['horizon_est'] - fine['horizon_est'])
                if (coarse['horizon_est'] is not None and fine['horizon_est'] is not None) else "NA"),
            hc=coarse["holes"], hf=fine["holes"],
            ec=(coarse["edge_blur"] if coarse["edge_blur"] is not None else "NA"),
            ef=(fine["edge_blur"]   if fine["edge_blur"]   is not None else "NA"),
            ic=coarse["interior_ok"], if_=fine["interior_ok"],
            mk=mesh_ok
        )
    )
    write_text(os.path.join(root, "outputs", "run_info", "result_line.txt"), result_line)
    print(result_line)

if __name__ == "__main__":
    main()
